<?php
include 'db.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $action = $_GET['action'] ?? '';
    
    if ($action == 'get_products') {
        $showInactive = $_GET['show_inactive'] ?? false;
        
        $sql = "SELECT p.*, s.current_qty 
                FROM products p 
                LEFT JOIN stock s ON p.id = s.product_id 
                WHERE 1=1";
        
        if (!$showInactive) {
            $sql .= " AND p.status = 'active'";
        }
        
        $sql .= " ORDER BY p.created_at DESC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $products = $stmt->fetchAll();
        
        $html = '';
        foreach ($products as $product) {
            $statusBadge = $product['status'] == 'active' 
                ? '<span class="badge bg-success">Active</span>' 
                : '<span class="badge bg-secondary">Inactive</span>';
            
            $html .= '<tr>';
            $html .= '<td>' . $product['id'] . '</td>';
            $html .= '<td><strong>' . htmlspecialchars($product['name']) . '</strong></td>';
            $html .= '<td>' . ($product['sku'] ? htmlspecialchars($product['sku']) : '<span class="text-muted">N/A</span>') . '</td>';
            $html .= '<td>' . ($product['category'] ? htmlspecialchars($product['category']) : '<span class="text-muted">N/A</span>') . '</td>';
            $html .= '<td class="text-danger">৳ ' . number_format($product['cost_price'], 2) . '</td>';
            $html .= '<td class="text-success">৳ ' . number_format($product['selling_price'], 2) . '</td>';
            $html .= '<td><span class="badge bg-info">' . ($product['current_qty'] ?? 0) . '</span></td>';
            $html .= '<td>' . $statusBadge . '</td>';
            $html .= '<td>' . date('M j, Y', strtotime($product['created_at'])) . '</td>';
            $html .= '<td>';
            $html .= '<div class="btn-group btn-group-sm">';
            $html .= '<button class="btn btn-outline-info view-product" data-id="' . $product['id'] . '" title="View Details">';
            $html .= '<i class="fas fa-eye"></i>';
            $html .= '</button>';
            $html .= '<button class="btn btn-outline-warning edit-product" data-id="' . $product['id'] . '" title="Edit">';
            $html .= '<i class="fas fa-edit"></i>';
            $html .= '</button>';
            $html .= '<button class="btn btn-outline-danger delete-product" data-id="' . $product['id'] . '" data-name="' . htmlspecialchars($product['name']) . '" title="Delete">';
            $html .= '<i class="fas fa-trash"></i>';
            $html .= '</button>';
            $html .= '</div>';
            $html .= '</td>';
            $html .= '</tr>';
        }
        
        if (empty($html)) {
            $html = '<tr><td colspan="10" class="text-center text-muted py-4">No products found. <a href="#" data-bs-toggle="modal" data-bs-target="#addProductModal">Add your first product</a></td></tr>';
        }
        
        echo $html;
    }
    elseif ($action == 'get_product') {
        $id = $_GET['id'];
        
        $stmt = $pdo->prepare("SELECT p.*, s.current_qty FROM products p LEFT JOIN stock s ON p.id = s.product_id WHERE p.id = ?");
        $stmt->execute([$id]);
        $product = $stmt->fetch();
        
        echo json_encode($product);
    }
    elseif ($action == 'get_product_details') {
        $id = $_GET['id'];
        
        // Get product details
        $stmt = $pdo->prepare("SELECT p.*, s.current_qty FROM products p LEFT JOIN stock s ON p.id = s.product_id WHERE p.id = ?");
        $stmt->execute([$id]);
        $product = $stmt->fetch();
        
        // Get recent sales
        $stmt = $pdo->prepare("SELECT * FROM sales WHERE product_id = ? ORDER BY sale_date DESC LIMIT 5");
        $stmt->execute([$id]);
        $sales = $stmt->fetchAll();
        
        echo json_encode([
            'product' => $product,
            'sales' => $sales
        ]);
    }
    elseif ($action == 'get_active_products') {
        $stmt = $pdo->prepare("SELECT * FROM products WHERE status = 'active' ORDER BY name");
        $stmt->execute();
        $products = $stmt->fetchAll();
        
        echo json_encode($products);
    }
}
elseif ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action == 'add_product') {
        try {
            $pdo->beginTransaction();
            
            $name = sanitize($_POST['name']);
            $sku = sanitize($_POST['sku']);
            $category = sanitize($_POST['category']);
            $cost_price = floatval($_POST['cost_price']);
            $selling_price = floatval($_POST['selling_price']);
            $initial_stock = intval($_POST['initial_stock']);
            $status = $_POST['status'] ?? 'active';
            
            // Generate SKU if empty
            if (empty($sku)) {
                $sku = strtoupper(preg_replace('/[^A-Z0-9]/', '', $name)) . rand(1000, 9999);
            }
            
            // Check if SKU already exists
            $stmt = $pdo->prepare("SELECT id FROM products WHERE sku = ?");
            $stmt->execute([$sku]);
            if ($stmt->fetch()) {
                throw new Exception("SKU already exists. Please use a different SKU.");
            }
            
            // Insert product
            $stmt = $pdo->prepare("INSERT INTO products (name, sku, category, cost_price, selling_price, status) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$name, $sku, $category, $cost_price, $selling_price, $status]);
            $product_id = $pdo->lastInsertId();
            
            // Insert stock record
            $stmt = $pdo->prepare("INSERT INTO stock (product_id, current_qty) VALUES (?, ?)");
            $stmt->execute([$product_id, $initial_stock]);
            
            // Record stock movement if initial stock > 0
            if ($initial_stock > 0) {
                $stmt = $pdo->prepare("INSERT INTO stock_movements (product_id, movement_type, quantity, prev_stock, new_stock, remarks) VALUES (?, 'adjustment', ?, 0, ?, 'Initial stock')");
                $stmt->execute([$product_id, $initial_stock, $initial_stock]);
            }
            
            $pdo->commit();
            
            echo json_encode(['success' => true, 'message' => 'Product added successfully!']);
        } catch (Exception $e) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
    elseif ($action == 'update_product') {
        try {
            $id = intval($_POST['id']);
            $name = sanitize($_POST['name']);
            $sku = sanitize($_POST['sku']);
            $current_sku = sanitize($_POST['current_sku']);
            $category = sanitize($_POST['category']);
            $cost_price = floatval($_POST['cost_price']);
            $selling_price = floatval($_POST['selling_price']);
            $status = $_POST['status'] ?? 'active';
            
            // Check if SKU is being changed and if new SKU already exists
            if ($sku != $current_sku) {
                $stmt = $pdo->prepare("SELECT id FROM products WHERE sku = ? AND id != ?");
                $stmt->execute([$sku, $id]);
                if ($stmt->fetch()) {
                    throw new Exception("SKU already exists. Please use a different SKU.");
                }
            }
            
            $stmt = $pdo->prepare("UPDATE products SET name = ?, sku = ?, category = ?, cost_price = ?, selling_price = ?, status = ? WHERE id = ?");
            $stmt->execute([$name, $sku, $category, $cost_price, $selling_price, $status, $id]);
            
            echo json_encode(['success' => true, 'message' => 'Product updated successfully!']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
    elseif ($action == 'delete_product') {
        try {
            $id = intval($_POST['id']);
            
            // Check if product has sales records
            $stmt = $pdo->prepare("SELECT COUNT(*) as sales_count FROM sales WHERE product_id = ?");
            $stmt->execute([$id]);
            $result = $stmt->fetch();
            
            if ($result['sales_count'] > 0) {
                throw new Exception("Cannot delete product. It has sales records associated with it. You can deactivate the product instead.");
            }
            
            // Check if product has stock movements
            $stmt = $pdo->prepare("SELECT COUNT(*) as movements_count FROM stock_movements WHERE product_id = ?");
            $stmt->execute([$id]);
            $result = $stmt->fetch();
            
            if ($result['movements_count'] > 0) {
                throw new Exception("Cannot delete product. It has stock history. You can deactivate the product instead.");
            }
            
            // Delete product (this will cascade to stock table due to foreign key)
            $stmt = $pdo->prepare("DELETE FROM products WHERE id = ?");
            $stmt->execute([$id]);
            
            echo json_encode(['success' => true, 'message' => 'Product deleted successfully!']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
}
?>