<?php 
include 'header.php';

// Handle AJAX requests
if (isset($_GET['action']) || isset($_POST['action'])) {
    $action = $_GET['action'] ?? $_POST['action'] ?? '';
    
    switch($action) {
        case 'get_active_products':
            getActiveProducts();
            break;
        case 'get_sales':
            getSales();
            break;
        case 'get_sale':
            getSale();
            break;
        case 'add_sale':
            addSale();
            break;
        case 'update_sale':
            updateSale();
            break;
        case 'delete_sale':
            deleteSale();
            break;
    }
    exit;
}

function getActiveProducts() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("SELECT id, name, cost_price, selling_price FROM products WHERE status = 'active' ORDER BY name");
        $stmt->execute();
        $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($products);
    } catch(PDOException $e) {
        echo json_encode([]);
    }
}

function getSales() {
    global $pdo;
    
    $date_from = $_GET['date_from'] ?? '';
    $date_to = $_GET['date_to'] ?? '';
    $product_id = $_GET['product_id'] ?? '';
    
    $sql = "SELECT s.*, p.name as product_name 
            FROM sales s 
            LEFT JOIN products p ON s.product_id = p.id 
            WHERE 1=1";
    
    $params = [];
    
    if (!empty($date_from)) {
        $sql .= " AND s.sale_date >= ?";
        $params[] = $date_from;
    }
    
    if (!empty($date_to)) {
        $sql .= " AND s.sale_date <= ?";
        $params[] = $date_to;
    }
    
    if (!empty($product_id)) {
        $sql .= " AND s.product_id = ?";
        $params[] = $product_id;
    }
    
    $sql .= " ORDER BY s.sale_date DESC, s.id DESC";
    
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $sales = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($sales)) {
            echo '<tr><td colspan="9" class="text-center">No sales records found</td></tr>';
            return;
        }
        
        foreach($sales as $sale) {
            $total_sale = $sale['quantity'] * $sale['selling_price'];
            $total_cost = $sale['quantity'] * $sale['cost_price'];
            $profit = $total_sale - $total_cost;
            $profit_percentage = $total_cost > 0 ? ($profit / $total_cost) * 100 : 0;
            
            echo "<tr>
                    <td>{$sale['sale_date']}</td>
                    <td>{$sale['product_name']}</td>
                    <td>{$sale['quantity']}</td>
                    <td>$" . number_format($sale['selling_price'], 2) . "</td>
                    <td>$" . number_format($total_sale, 2) . "</td>
                    <td>$" . number_format($total_cost, 2) . "</td>
                    <td>$" . number_format($profit, 2) . "</td>
                    <td>" . number_format($profit_percentage, 2) . "%</td>
                    <td>
                        <button class='btn btn-sm btn-outline-primary edit-sale' data-id='{$sale['id']}'>
                            <i class='fas fa-edit'></i>
                        </button>
                        <button class='btn btn-sm btn-outline-danger delete-sale' data-id='{$sale['id']}' data-name='{$sale['product_name']}'>
                            <i class='fas fa-trash'></i>
                        </button>
                    </td>
                  </tr>";
        }
    } catch(PDOException $e) {
        echo '<tr><td colspan="9" class="text-center text-danger">Error loading sales data</td></tr>';
    }
}

function getSale() {
    global $pdo;
    $id = $_GET['id'] ?? 0;
    
    if (!$id) {
        echo json_encode(null);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM sales WHERE id = ?");
        $stmt->execute([$id]);
        $sale = $stmt->fetch(PDO::FETCH_ASSOC);
        echo json_encode($sale);
    } catch(PDOException $e) {
        echo json_encode(null);
    }
}

function addSale() {
    global $pdo;
    
    $product_id = $_POST['product_id'] ?? 0;
    $sale_date = $_POST['sale_date'] ?? '';
    $quantity = $_POST['quantity'] ?? 0;
    $selling_price = $_POST['selling_price'] ?? 0;
    $cost_price = $_POST['cost_price'] ?? 0;
    $remarks = $_POST['remarks'] ?? '';
    
    if (!$product_id || !$sale_date || $quantity <= 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid input data']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("INSERT INTO sales (product_id, sale_date, quantity, selling_price, cost_price, remarks) 
                              VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$product_id, $sale_date, $quantity, $selling_price, $cost_price, $remarks]);
        
        echo json_encode(['success' => true, 'message' => 'Sale added successfully']);
    } catch(PDOException $e) {
        echo json_encode(['success' => false, 'message' => 'Error adding sale: ' . $e->getMessage()]);
    }
}

function updateSale() {
    global $pdo;
    
    $id = $_POST['sale_id'] ?? 0;
    $product_id = $_POST['product_id'] ?? 0;
    $sale_date = $_POST['sale_date'] ?? '';
    $quantity = $_POST['quantity'] ?? 0;
    $selling_price = $_POST['selling_price'] ?? 0;
    $cost_price = $_POST['cost_price'] ?? 0;
    $remarks = $_POST['remarks'] ?? '';
    
    if (!$id || !$product_id || !$sale_date || $quantity <= 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid input data']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("UPDATE sales SET product_id = ?, sale_date = ?, quantity = ?, 
                              selling_price = ?, cost_price = ?, remarks = ? WHERE id = ?");
        $stmt->execute([$product_id, $sale_date, $quantity, $selling_price, $cost_price, $remarks, $id]);
        
        echo json_encode(['success' => true, 'message' => 'Sale updated successfully']);
    } catch(PDOException $e) {
        echo json_encode(['success' => false, 'message' => 'Error updating sale: ' . $e->getMessage()]);
    }
}

function deleteSale() {
    global $pdo;
    
    $id = $_POST['id'] ?? 0;
    
    if (!$id) {
        echo json_encode(['success' => false, 'message' => 'Invalid sale ID']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("DELETE FROM sales WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['success' => true, 'message' => 'Sale deleted successfully']);
    } catch(PDOException $e) {
        echo json_encode(['success' => false, 'message' => 'Error deleting sale: ' . $e->getMessage()]);
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1 class="h3 mb-0"><i class="fas fa-shopping-cart me-2"></i> Sales Management</h1>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addSaleModal">
                <i class="fas fa-plus me-1"></i> Add Sale
            </button>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card shadow mb-4">
    <div class="card-body">
        <form id="salesFilterForm">
            <div class="row">
                <div class="col-md-3">
                    <label class="form-label">Date From</label>
                    <input type="date" class="form-control" name="date_from" id="dateFrom">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Date To</label>
                    <input type="date" class="form-control" name="date_to" id="dateTo">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Product</label>
                    <select class="form-select" name="product_id" id="productFilter">
                        <option value="">All Products</option>
                    </select>
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="button" class="btn btn-outline-primary me-2" id="applyFilters">Apply</button>
                    <button type="button" class="btn btn-outline-secondary" id="resetFilters">Reset</button>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Sales Table -->
<div class="card shadow">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Sales Records</h6>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-bordered table-hover" id="salesTable" width="100%" cellspacing="0">
                <thead class="table-light">
                    <tr>
                        <th>Date</th>
                        <th>Product</th>
                        <th>Quantity</th>
                        <th>Unit Price</th>
                        <th>Total Sale</th>
                        <th>Cost</th>
                        <th>Profit</th>
                        <th>Profit %</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <!-- Data will be loaded via AJAX -->
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add Sale Modal -->
<div class="modal fade" id="addSaleModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Sale</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="addSaleForm">
                    <input type="hidden" name="sale_id" id="saleId">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Product</label>
                                <select class="form-select" name="product_id" id="saleProduct" required>
                                    <option value="">Select Product</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Sale Date</label>
                                <input type="date" class="form-control" name="sale_date" value="<?php echo date('Y-m-d'); ?>" required>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Quantity</label>
                                <input type="number" class="form-control" name="quantity" min="1" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Selling Price</label>
                                <input type="number" class="form-control" name="selling_price" step="0.01" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Cost Price</label>
                                <input type="number" class="form-control" name="cost_price" step="0.01" required>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Remarks (Optional)</label>
                        <textarea class="form-control" name="remarks" rows="2"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveSale">Save Sale</button>
            </div>
        </div>
    </div>
</div>

<!-- Alert Container -->
<div id="alertContainer" style="position: fixed; top: 20px; right: 20px; z-index: 9999;"></div>

<script>
$(document).ready(function() {
    // Alert function
    function showAlert(message, type = 'success') {
        const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
        const alertHtml = `
            <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;
        $('#alertContainer').append(alertHtml);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            $('.alert').alert('close');
        }, 5000);
    }
    
    // Load products for dropdowns
    function loadProducts() {
        $.ajax({
            url: window.location.href,
            type: 'GET',
            data: { action: 'get_active_products' },
            success: function(response) {
                try {
                    const products = JSON.parse(response);
                    let options = '<option value="">All Products</option>';
                    let saleOptions = '<option value="">Select Product</option>';
                    
                    products.forEach(product => {
                        options += `<option value="${product.id}">${product.name}</option>`;
                        saleOptions += `<option value="${product.id}" data-cost="${product.cost_price}" data-price="${product.selling_price}">${product.name}</option>`;
                    });
                    
                    $('#productFilter').html(options);
                    $('#saleProduct').html(saleOptions);
                } catch (e) {
                    console.error('Error parsing products:', e);
                    showAlert('Error loading products', 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading products:', error);
                showAlert('Error loading products', 'error');
            }
        });
    }
    
    // Load sales data
    function loadSales() {
        const formData = $('#salesFilterForm').serialize();
        
        $.ajax({
            url: window.location.href,
            type: 'GET',
            data: formData + '&action=get_sales',
            success: function(response) {
                $('#salesTable tbody').html(response);
            },
            error: function(xhr, status, error) {
                console.error('Error loading sales:', error);
                showAlert('Error loading sales data', 'error');
            }
        });
    }
    
    // Product change in sale form
    $('#saleProduct').change(function() {
        const selectedOption = $(this).find('option:selected');
        const costPrice = selectedOption.data('cost');
        const sellingPrice = selectedOption.data('price');
        
        if (costPrice) {
            $('input[name="cost_price"]').val(costPrice);
        }
        if (sellingPrice) {
            $('input[name="selling_price"]').val(sellingPrice);
        }
    });
    
    // Save sale (both add and edit)
    $('#saveSale').click(function() {
        const formData = $('#addSaleForm').serialize();
        const saleId = $('#saleId').val();
        const action = saleId ? 'update_sale' : 'add_sale';
        
        $.ajax({
            url: window.location.href,
            type: 'POST',
            data: formData + '&action=' + action,
            success: function(response) {
                try {
                    const result = JSON.parse(response);
                    if (result.success) {
                        $('#addSaleModal').modal('hide');
                        $('#addSaleForm')[0].reset();
                        $('#saleId').val('');
                        loadSales();
                        showAlert(result.message || 'Sale saved successfully!', 'success');
                    } else {
                        showAlert(result.message || 'Error saving sale', 'error');
                    }
                } catch (e) {
                    console.error('Error parsing response:', e);
                    showAlert('Error processing response', 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error saving sale:', error);
                showAlert('Error saving sale', 'error');
            }
        });
    });
    
    // Edit sale
    $(document).on('click', '.edit-sale', function() {
        const saleId = $(this).data('id');
        
        $.ajax({
            url: window.location.href,
            type: 'GET',
            data: { action: 'get_sale', id: saleId },
            success: function(response) {
                try {
                    const sale = JSON.parse(response);
                    if (sale) {
                        $('#saleId').val(sale.id);
                        $('input[name="sale_date"]').val(sale.sale_date);
                        $('select[name="product_id"]').val(sale.product_id);
                        $('input[name="quantity"]').val(sale.quantity);
                        $('input[name="selling_price"]').val(sale.selling_price);
                        $('input[name="cost_price"]').val(sale.cost_price);
                        $('textarea[name="remarks"]').val(sale.remarks || '');
                        
                        $('#addSaleModal .modal-title').text('Edit Sale');
                        $('#addSaleModal').modal('show');
                    }
                } catch (e) {
                    console.error('Error parsing sale data:', e);
                    showAlert('Error loading sale data', 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading sale:', error);
                showAlert('Error loading sale data', 'error');
            }
        });
    });
    
    // Delete sale
    $(document).on('click', '.delete-sale', function() {
        const saleId = $(this).data('id');
        const productName = $(this).data('name');
        
        if (confirm(`Are you sure you want to delete sale for ${productName}?`)) {
            $.ajax({
                url: window.location.href,
                type: 'POST',
                data: { action: 'delete_sale', id: saleId },
                success: function(response) {
                    try {
                        const result = JSON.parse(response);
                        if (result.success) {
                            loadSales();
                            showAlert(result.message || 'Sale deleted successfully!', 'success');
                        } else {
                            showAlert(result.message || 'Error deleting sale', 'error');
                        }
                    } catch (e) {
                        console.error('Error parsing response:', e);
                        showAlert('Error processing response', 'error');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error deleting sale:', error);
                    showAlert('Error deleting sale', 'error');
                }
            });
        }
    });
    
    // Apply filters
    $('#applyFilters').click(function() {
        loadSales();
    });
    
    // Reset filters
    $('#resetFilters').click(function() {
        $('#salesFilterForm')[0].reset();
        loadSales();
    });
    
    // Reset modal when closed
    $('#addSaleModal').on('hidden.bs.modal', function() {
        $('#addSaleForm')[0].reset();
        $('#saleId').val('');
        $('#addSaleModal .modal-title').text('Add New Sale');
    });
    
    // Initialize
    loadProducts();
    loadSales();
    
    // Set default dates
    const today = new Date().toISOString().split('T')[0];
    const lastMonth = new Date();
    lastMonth.setMonth(lastMonth.getMonth() - 1);
    const lastMonthStr = lastMonth.toISOString().split('T')[0];
    
    $('#dateFrom').val(lastMonthStr);
    $('#dateTo').val(today);
});
</script>