// Custom JavaScript for Tally System

// Global function to show alerts
function showAlert(message, type = 'info') {
    const alertClass = {
        'success': 'alert-success',
        'error': 'alert-danger',
        'warning': 'alert-warning',
        'info': 'alert-info'
    }[type] || 'alert-info';
    
    const alertHtml = `
        <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    // Create container if it doesn't exist
    if (!$('#alertContainer').length) {
        $('.container-fluid').prepend('<div id="alertContainer"></div>');
    }
    
    $('#alertContainer').html(alertHtml);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        $('.alert').alert('close');
    }, 5000);
}

// Format currency
function formatCurrency(amount) {
    return '৳ ' + parseFloat(amount).toLocaleString('en-US', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

// Format percentage
function formatPercent(value) {
    return parseFloat(value).toFixed(2) + '%';
}

// Date utilities
function getDateRange(period) {
    const today = new Date();
    const start = new Date();
    
    switch(period) {
        case 'today':
            start.setHours(0, 0, 0, 0);
            return {
                start: start,
                end: today
            };
        case 'yesterday':
            start.setDate(today.getDate() - 1);
            start.setHours(0, 0, 0, 0);
            const yesterdayEnd = new Date(start);
            yesterdayEnd.setHours(23, 59, 59, 999);
            return {
                start: start,
                end: yesterdayEnd
            };
        case 'last_week':
            start.setDate(today.getDate() - 7);
            return {
                start: start,
                end: today
            };
        case 'last_30':
            start.setDate(today.getDate() - 30);
            return {
                start: start,
                end: today
            };
        case 'last_month':
            start.setMonth(today.getMonth() - 1);
            return {
                start: start,
                end: today
            };
        case 'last_year':
            start.setFullYear(today.getFullYear() - 1);
            return {
                start: start,
                end: today
            };
        case 'all_time':
            return {
                start: new Date('2000-01-01'),
                end: today
            };
        default:
            return {
                start: start,
                end: today
            };
    }
}

// AJAX error handling
$(document).ajaxError(function(event, jqXHR, ajaxSettings, thrownError) {
    showAlert('An error occurred while processing your request. Please try again.', 'error');
    console.error('AJAX Error:', thrownError);
});

// Initialize tooltips
$(document).ready(function() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    const tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
});

// Auto-hide alerts
$(document).on('click', '.alert .btn-close', function() {
    $(this).closest('.alert').alert('close');
});

// Confirm before destructive actions
$(document).on('click', '[data-confirm]', function(e) {
    const message = $(this).data('confirm') || 'Are you sure?';
    if (!confirm(message)) {
        e.preventDefault();
        e.stopPropagation();
        return false;
    }
});

// Form validation enhancement
$(document).on('submit', 'form', function(e) {
    const form = $(this);
    const requiredFields = form.find('[required]');
    let valid = true;
    
    requiredFields.each(function() {
        if (!$(this).val().trim()) {
            $(this).addClass('is-invalid');
            valid = false;
        } else {
            $(this).removeClass('is-invalid');
        }
    });
    
    if (!valid) {
        e.preventDefault();
        showAlert('Please fill in all required fields.', 'warning');
    }
});

// Real-time form validation
$(document).on('blur', '[required]', function() {
    if (!$(this).val().trim()) {
        $(this).addClass('is-invalid');
    } else {
        $(this).removeClass('is-invalid');
    }
});

// Number formatting
$(document).on('blur', 'input[type="number"]', function() {
    const value = parseFloat($(this).val());
    if (!isNaN(value)) {
        $(this).val(value.toFixed(2));
    }
});

// Auto-calculate total
$(document).on('input', '.calc-total', function() {
    const row = $(this).closest('tr');
    const quantity = parseFloat(row.find('.quantity').val()) || 0;
    const price = parseFloat(row.find('.price').val()) || 0;
    const total = quantity * price;
    row.find('.total').val(total.toFixed(2));
});

// Mobile menu enhancement
$(document).on('click', '.navbar-toggler', function() {
    $('.navbar-collapse').toggleClass('show');
});

// Print functionality
function printElement(element) {
    const printContent = document.getElementById(element).innerHTML;
    const originalContent = document.body.innerHTML;
    
    document.body.innerHTML = printContent;
    window.print();
    document.body.innerHTML = originalContent;
    window.location.reload();
}

// Export to CSV
function exportToCSV(tableId, filename) {
    const table = document.getElementById(tableId);
    const rows = table.querySelectorAll('tr');
    const csv = [];
    
    for (let i = 0; i < rows.length; i++) {
        const row = [], cols = rows[i].querySelectorAll('td, th');
        
        for (let j = 0; j < cols.length; j++) {
            // Remove action columns
            if (!cols[j].closest('th') || !cols[j].querySelector('.btn')) {
                row.push(cols[j].innerText);
            }
        }
        
        csv.push(row.join(','));
    }
    
    // Download CSV file
    const csvFile = new Blob([csv.join('\n')], { type: 'text/csv' });
    const downloadLink = document.createElement('a');
    downloadLink.download = filename;
    downloadLink.href = window.URL.createObjectURL(csvFile);
    downloadLink.style.display = 'none';
    document.body.appendChild(downloadLink);
    downloadLink.click();
    document.body.removeChild(downloadLink);
}

// Theme toggle (for future dark mode)
function toggleTheme() {
    document.body.classList.toggle('dark-theme');
    localStorage.setItem('theme', document.body.classList.contains('dark-theme') ? 'dark' : 'light');
}

// Load saved theme
$(document).ready(function() {
    const savedTheme = localStorage.getItem('theme');
    if (savedTheme === 'dark') {
        document.body.classList.add('dark-theme');
    }
});