<?php 
include 'header.php';

// Handle Form Submissions FIRST before any HTML output
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_product'])) {
        try {
            $pdo->beginTransaction();
            
            $name = sanitize($_POST['name']);
            $cost_price = floatval($_POST['cost_price']);
            $selling_price = floatval($_POST['selling_price']);
            $initial_stock = intval($_POST['initial_stock']);
            $status = isset($_POST['status']) ? 'active' : 'inactive';
            
            // Insert product
            $stmt = $pdo->prepare("INSERT INTO products (name, cost_price, selling_price, status) VALUES (?, ?, ?, ?)");
            $stmt->execute([$name, $cost_price, $selling_price, $status]);
            $product_id = $pdo->lastInsertId();
            
            // Insert stock record
            $stmt = $pdo->prepare("INSERT INTO stock (product_id, current_qty) VALUES (?, ?)");
            $stmt->execute([$product_id, $initial_stock]);
            
            // Record stock movement if initial stock > 0
            if ($initial_stock > 0) {
                $stmt = $pdo->prepare("INSERT INTO stock_movements (product_id, movement_type, quantity, prev_stock, new_stock, remarks) VALUES (?, 'adjustment', ?, 0, ?, 'Initial stock')");
                $stmt->execute([$product_id, $initial_stock, $initial_stock]);
            }
            
            $pdo->commit();
            
            $_SESSION['success'] = 'Product added successfully!';
            header("Location: products.php");
            exit;
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $_SESSION['error'] = $e->getMessage();
            header("Location: products.php");
            exit;
        }
    }
    elseif (isset($_POST['update_product'])) {
        try {
            $id = intval($_POST['id']);
            $name = sanitize($_POST['name']);
            $cost_price = floatval($_POST['cost_price']);
            $selling_price = floatval($_POST['selling_price']);
            $status = isset($_POST['status']) ? 'active' : 'inactive';
            
            $stmt = $pdo->prepare("UPDATE products SET name = ?, cost_price = ?, selling_price = ?, status = ? WHERE id = ?");
            $stmt->execute([$name, $cost_price, $selling_price, $status, $id]);
            
            $_SESSION['success'] = 'Product updated successfully!';
            header("Location: products.php");
            exit;
            
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header("Location: products.php");
            exit;
        }
    }
    elseif (isset($_POST['delete_product'])) {
        try {
            $id = intval($_POST['id']);
            
            // Check if product has sales records
            $stmt = $pdo->prepare("SELECT COUNT(*) as sales_count FROM sales WHERE product_id = ?");
            $stmt->execute([$id]);
            $result = $stmt->fetch();
            
            if ($result['sales_count'] > 0) {
                throw new Exception("Cannot delete product. It has sales records associated with it. You can deactivate the product instead.");
            }
            
            // Check if product has stock movements
            $stmt = $pdo->prepare("SELECT COUNT(*) as movements_count FROM stock_movements WHERE product_id = ?");
            $stmt->execute([$id]);
            $result = $stmt->fetch();
            
            if ($result['movements_count'] > 0) {
                throw new Exception("Cannot delete product. It has stock history. You can deactivate the product instead.");
            }
            
            // Delete product
            $stmt = $pdo->prepare("DELETE FROM products WHERE id = ?");
            $stmt->execute([$id]);
            
            $_SESSION['success'] = 'Product deleted successfully!';
            header("Location: products.php");
            exit;
            
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header("Location: products.php");
            exit;
        }
    }
}

// Get all products
$showInactive = isset($_GET['show_inactive']) ? true : false;

$sql = "SELECT p.*, s.current_qty 
        FROM products p 
        LEFT JOIN stock s ON p.id = s.product_id 
        WHERE 1=1";

if (!$showInactive) {
    $sql .= " AND p.status = 'active'";
}

$sql .= " ORDER BY p.created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute();
$products = $stmt->fetchAll();
?>

<!-- Main Content -->
<div class="container-fluid mt-4">
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <h1 class="h3 mb-0"><i class="fas fa-boxes me-2"></i> Product Management</h1>
                <button class="btn btn-primary" type="button" data-bs-toggle="modal" data-bs-target="#addProductModal">
                    <i class="fas fa-plus me-1"></i> Add Product
                </button>
            </div>
        </div>
    </div>

    <!-- Display Messages -->
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle me-2"></i><?php echo $_SESSION['success']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle me-2"></i><?php echo $_SESSION['error']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <!-- Products Table -->
    <div class="card shadow">
        <div class="card-header py-3 d-flex justify-content-between align-items-center">
            <h6 class="m-0 font-weight-bold text-primary">All Products</h6>
            <div class="form-check form-switch">
                <input class="form-check-input" type="checkbox" id="showInactive" <?php echo $showInactive ? 'checked' : ''; ?> onchange="toggleInactiveProducts()">
                <label class="form-check-label" for="showInactive">Show Inactive</label>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-hover" id="productsTable" width="100%" cellspacing="0">
                    <thead class="table-light">
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Cost Price</th>
                            <th>Selling Price</th>
                            <th>Stock</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (count($products) > 0): ?>
                            <?php foreach ($products as $product): ?>
                                <?php
                                $statusBadge = $product['status'] == 'active' 
                                    ? '<span class="badge bg-success">Active</span>' 
                                    : '<span class="badge bg-secondary">Inactive</span>';
                                ?>
                                <tr>
                                    <td><?php echo $product['id']; ?></td>
                                    <td><strong><?php echo htmlspecialchars($product['name']); ?></strong></td>
                                    <td class="text-danger">৳ <?php echo number_format($product['cost_price'], 2); ?></td>
                                    <td class="text-success">৳ <?php echo number_format($product['selling_price'], 2); ?></td>
                                    <td><span class="badge bg-info"><?php echo $product['current_qty'] ?? 0; ?></span></td>
                                    <td><?php echo $statusBadge; ?></td>
                                    <td><?php echo date('M j, Y', strtotime($product['created_at'])); ?></td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-outline-warning edit-product" data-bs-toggle="modal" data-bs-target="#editProductModal" 
                                                    data-id="<?php echo $product['id']; ?>"
                                                    data-name="<?php echo htmlspecialchars($product['name']); ?>"
                                                    data-cost="<?php echo $product['cost_price']; ?>"
                                                    data-price="<?php echo $product['selling_price']; ?>"
                                                    data-status="<?php echo $product['status']; ?>">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete <?php echo htmlspecialchars($product['name']); ?>?')">
                                                <input type="hidden" name="id" value="<?php echo $product['id']; ?>">
                                                <button type="submit" name="delete_product" class="btn btn-outline-danger">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="8" class="text-center text-muted py-4">
                                    No products found. <a href="#" data-bs-toggle="modal" data-bs-target="#addProductModal">Add your first product</a>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Add Product Modal -->
<div class="modal fade" id="addProductModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title">
                    <i class="fas fa-plus-circle me-2"></i>Add New Product
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="products.php">
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Product Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" name="name" required placeholder="Enter product name">
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Cost Price (৳) <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">৳</span>
                                    <input type="number" class="form-control" name="cost_price" step="0.01" min="0" required>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Selling Price (৳) <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">৳</span>
                                    <input type="number" class="form-control" name="selling_price" step="0.01" min="0" required>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Initial Stock</label>
                        <input type="number" class="form-control" name="initial_stock" value="0" min="0">
                    </div>
                    
                    <div class="mb-3">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" name="status" value="active" checked>
                            <label class="form-check-label">Product Active</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_product" class="btn btn-primary">Save Product</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Product Modal -->
<div class="modal fade" id="editProductModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title">
                    <i class="fas fa-edit me-2"></i>Edit Product
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="products.php">
                <input type="hidden" name="id" id="editProductId">
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Product Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" name="name" id="editProductName" required>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Cost Price (৳) <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">৳</span>
                                    <input type="number" class="form-control" name="cost_price" id="editProductCost" step="0.01" min="0" required>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Selling Price (৳) <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">৳</span>
                                    <input type="number" class="form-control" name="selling_price" id="editProductPrice" step="0.01" min="0" required>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" name="status" value="active" id="editProductStatus">
                            <label class="form-check-label">Product Active</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="update_product" class="btn btn-warning">Update Product</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Toggle inactive products
function toggleInactiveProducts() {
    const showInactive = document.getElementById('showInactive').checked;
    window.location.href = 'products.php?show_inactive=' + (showInactive ? '1' : '0');
}

// Edit product modal functionality
document.addEventListener('DOMContentLoaded', function() {
    const editButtons = document.querySelectorAll('.edit-product');
    
    editButtons.forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-id');
            const productName = this.getAttribute('data-name');
            const costPrice = this.getAttribute('data-cost');
            const sellingPrice = this.getAttribute('data-price');
            const status = this.getAttribute('data-status');
            
            document.getElementById('editProductId').value = productId;
            document.getElementById('editProductName').value = productName;
            document.getElementById('editProductCost').value = costPrice;
            document.getElementById('editProductPrice').value = sellingPrice;
            document.getElementById('editProductStatus').checked = status === 'active';
        });
    });
    
    // Price validation
    const addForm = document.querySelector('#addProductModal form');
    const editForm = document.querySelector('#editProductModal form');
    
    if (addForm) {
        addForm.addEventListener('submit', function(e) {
            const costPrice = parseFloat(this.querySelector('input[name="cost_price"]').value) || 0;
            const sellingPrice = parseFloat(this.querySelector('input[name="selling_price"]').value) || 0;
            
            if (sellingPrice <= costPrice) {
                e.preventDefault();
                alert('Selling price should be higher than cost price for profit!');
                return false;
            }
            return true;
        });
    }
    
    if (editForm) {
        editForm.addEventListener('submit', function(e) {
            const costPrice = parseFloat(this.querySelector('input[name="cost_price"]').value) || 0;
            const sellingPrice = parseFloat(this.querySelector('input[name="selling_price"]').value) || 0;
            
            if (sellingPrice <= costPrice) {
                e.preventDefault();
                alert('Selling price should be higher than cost price for profit!');
                return false;
            }
            return true;
        });
    }
    
    // Auto-focus on first input when modal opens
    const addModal = document.getElementById('addProductModal');
    if (addModal) {
        addModal.addEventListener('shown.bs.modal', function() {
            this.querySelector('input[name="name"]').focus();
        });
    }
    
    const editModal = document.getElementById('editProductModal');
    if (editModal) {
        editModal.addEventListener('shown.bs.modal', function() {
            this.querySelector('input[name="name"]').focus();
        });
    }
});
</script>

<?php include 'footer.php'; ?>